' Copyright 2007 BFF Design Ltd
'
' Ian Hopper (BFF Design Ltd) Sept 2007
' For 3 DOF DIY Motion Cockpit
' Ian@Builtforfun.co.uk
' http://buggies.builtforfun.co.uk/
'
'=====================================================
' Pos2SpeedRoll.bas is free software; you can redistribute it and/or modify
' it under the terms of the GNU General Public License as published by
' the Free Software Foundation; either version 3 of the License, or
' (at your option) any later version.
'
' Pos2SpeedRoll.bas is distributed in the hope that it will be useful,
' but WITHOUT ANY WARRANTY; without even the implied warranty of
' MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
' GNU General Public License for more details.
'
' A copy of the GNU General Public License is included in this zip archive
' You can also see it at http://www.gnu.org/licenses/.
'====================================================
'
' PICAXE program to convert a position demand into a pwm 0 to 5V output speed demand.
'
' Pin Designations
'
'	Out0 	High for reverse
'	PWM3	Out3 0-5V PWM Speed Out
'	Out4	Ready to receive
'	Out7	Data receive LED
'	In6	Position Demand Input Pulse
'	In7	High for Data Available
'	In0	High for Slow Movement
'	ADC1	In1 0-5V Position Feedback
'
' Both this 18X and the sending 28X chip
' must be using 4 MHz resonators
'
' This version is for the ROLL Degree of Freedom  v1.0

'Variable Declarations

	symbol	Pdem = w6
	symbol	Pdif = w3
	symbol	Pact = w4
	symbol	Speed = w1
	symbol	pwmDuty = w5
	symbol	cutoff = w2
	symbol	temp = w0

'Set initial values

	Pdem = 1275		'ie midposition
	cutoff = 300 	'stroke length is about 800 
	pwmDuty = 40

'Switch any output on pin 3 to zero to ensure speed controller does not get an initial
'speed drive signal until we are ready. 

	switchoff 3
	switchoff 4
	
	

'Main loop to monitor position feedback and calculate and set speed output signals

MainLoop:


'Check if the data-ready pin is high, if so goto getdata and read the new position demand
'if not then continue with the current position demand

	if pin7 = 1 then getdata 

gotdata:

'Switch off the data receive LED

	switchoff 7

'We now have a position demand so read the current position and calculate the required
'speed output to take the DOF towards the demanded position.

	gosub setspeed

'ok, now go back to the beginning of the loop and start again

	goto MainLoop
	
	

'
'===============================================================
'

'Subroutine to set the demand speed and direction
setspeed:
	

'Read the current position

	readadc10 1, Pact

	
'If the pot is within 10% of its range it shouldn't be there - kill the drive signal
'This condition can happen if the pots fail.

	if Pact <= 100 or Pact >= 900 then 
	
		Speed = 0
		pwmDuty = 0
		pwmout 3, 255, pwmDuty
		end
	
	end if	

	Pact = Pact * 25 / 10 MAX 2550	'convert from 0 to 1024, to 0 to 2550

'If Pact is greater than Pdem then we need a negative speed to bring position back to that demanded

	if Pact > Pdem then gosub comeback

'Otherwise Pact is less than Pdem so we need a positive speed

	if Pact <= Pdem then gosub goforward
	
'Now we have the speed and the reverse pin has been set - send a pwm voltage to the speed controller
'The biggest speed can be is 255 and we want this to give 5V output.
	
	pwmDuty = Speed * 10 / 13  MAX 200 				'approx 5V MAX, (200 for 5V at 20kHz)
	
	if pin0 = 1 and pwmDuty > 50 then				'if the slow rate pin is high limit speed to 1/5 max
		pwmDuty = 50
	end if
	
'If required add in a small fixed voltage to overcome any modulation or movement threshold in the system - set this on test
'if it is too high the movement will oscillate.

	if pwmDuty < 12 then
		pwmDuty = 12
	end if
		
'and send the pwm signal to pwm output pin 3

	pwmout 3, 49, pwmDuty

	return

'===============================================================

'Demand speed for +ve movement
goforward:

	peek 80, b1					'Read current direction of travel flag
	peek 81, b0
	
	if temp = 0 then				'current direction is backwards - need to ramp down 1st
	
		gosub turnaround
		return
	
	else						'direction of travel is correct - so set speed as normal
	
		switchon 0
		temp = 1
		poke 80, b1
		poke 81, b0
	
		Pdif = Pdem - Pact
		if Pdif < cutoff then gosub near
		if Pdif >= cutoff then gosub far
		return
		
	end if
	
return

 
'Demand speed for -ve movement
comeback:

	peek 80, b1					'Read current direction of travel flag
	peek 81, b0
	
	if temp = 1 then				'current direction is forwards - need to ramp down 1st
	
		gosub turnaround
		return
	
	else						'direction of travel is correct - so set speed as normal

		switchoff 0
		temp = 0
		poke 80, b1
		poke 81, b0

		Pdif = Pact - Pdem
		if Pdif < cutoff then gosub near
		if Pdif >= cutoff then gosub far
		return
		
	end if
	
return

	
'Calculate speed when actual position is close to demanded position
near:

	temp = Pdif * 25 / cutoff * 10 MAX 254			'ie base speed tends to zero as Pdif tends to zero
	
	if temp < Speed then						'if actual speed is greater than the base speed then
		
		w5 = Speed - temp * 10 / 25 MIN 1 MAX 100
		Speed = Speed - w5 MIN 0				'ramp down the speed
		
	else	
		w5 = temp - Speed * 10 / 25 MIN 1 MAX 100			'Otherwise if the current speed is less than the base	
		Speed = Speed + w5 MAX 255				'ramp up the current speed
		
	end if
	
return


'Calculate speed when actual position is far from demanded position
far:

	Speed = Speed + 100 MAX 255					'ramp up the speed quickly

return


'Calculate speed when a direction reversal is required
turnaround:

	if Speed > 11 then
	
		w5 = Speed * 10 / 25 MIN 10 MAX 100
		Speed = Speed - w5 MIN 0				'ramp down the speed quickly to a min of zero
	
	else 									'if speed is now zero reverse direction
	
		peek 80, b1							'read current direction
		peek 81, b0
		
		if temp = 0 then						'if reverse switch to forward
		
			switchon 0
			temp = 1
			poke 80, b1
			poke 81, b0
		else
										'if forward switch to reverse
			switchoff 0
			temp = 0
			poke 80, b1
			poke 81, b0
			
		end if
		
	end if
		
return



'===============================================================


'Send the DOF to midposition 
gotomid:

	Pdem = 1275
	for temp = 1 to 100
		gosub setspeed
	next temp
	
'Back to the top of the main loop

return


'===============================================================


'Wait for then read the input pulse length
'Input pulse length is 10 to 265, the 1st 10 is dead 
getdata:

	Pdem = 137			'Mid position as default if pulsin times-out

	switchon 7			'Switch on the data receive LED
	
	pulsout 4,10		'send a short pulse on pin 4 to the 28X to initiate data transfer

	pulsin 6, 1, Pdem		'read input pulse length
	
	Pdem = Pdem - 10 * 10 MIN 1	'convert to position demand value
	
'Back to gotdata

goto gotdata  




	
	











































