' Copyright 2007 BFF Design Ltd
'
' Ian Hopper (BFF Design Ltd) Oct 2007
' http://buggies.builtforfun.co.uk - email Ian@builtforfun.co.uk
'
' For DIY 3 DOF Motion Cockpit Signal Processor Unit
'
'=====================================================
' ReadPosGeneric.bas has been provided free of charge WITHOUT ANY WARRANTY
' without even the implied warranty of MERCHANTABILITY or FITNESS FOR
' A PARTICULAR PURPOSE. You assume all responsibility for the consequences
' of your use of ReadPosGenericGeneric.bas.
' 
' ReadPosGeneric.bas is licensed for your personal NON-COMMERCIAL use only. I am
' happy to negotiate a reasonable charge with you if you wish to use ReadPosGeneric.bas
' for the purposes of commercial development including but not limited to research
' &  development, design feasibility study, prototype development, production and/or
' in clusion within or packaging with domestic or commercial products.
'
' You may not further distribute ReadPosGeneric.bas. 
'====================================================
'
' Read 3 position demand bytes from serial input on pin0 and forward on to the 3
' DOF position-to-speed demand coverter chips. Both this 28X and the 18X chips
' must be using 4 MHz resonators. This version also performs the initial set up of
' the SPU 18X chips.
'
' Version 2.0 Oct 07
'
' PICAXE 28X Pin designations
'
'	Out7	Pitch position output demand pulse
'	Out6	Pitch data-ready
'	Out5	Pitch high for slow rate
'
'	Out4	Roll position output demand pulse
'	Out3	Roll data-ready
'	Out2	Roll high for slow rate
'
'	Out1	Heave position output demand pulse
'	Out0	Heave data-ready
'	Outc7	Heave high for slow rate
'
'	In2	Heave 18X pulse for ready to receive
'	In3	Roll 18X pulse for ready to receive
'	In4	Pitch 18X pulse for ready to receive	
'
'	In0	Data Serial Input from PC
'

'Variable Declarations

	symbol	code = b0
	symbol	pos1 = b1
	symbol	pos2 = b2
	symbol	pos3 = b3
	symbol	ready = w3
	symbol	pulselength1 = w4
	symbol	pulselength2 = w5
	symbol	pulselength3 = w6


'Write the default 18X settings into EEPROM

	
	eeprom 1, (0, 127, 1, 44, 0, 5, 0, 127, 1, 44, 0, 12, 0, 155, 0, 200, 0, 20)	
	'equivalent to words (127,300,5,127,300,12,155,200,20) but in byte format 
	
	
'pos1 is the pitch DOF, pos2 the roll DOF and pos3 the heave DOF

'Set the start values

	dirsc = %10000000		'set pin c7 to be output pin for the heave DOF	

	switchoff 5
	switchoff 2
	switchoff portc 7
	
	switchoff 6
	switchoff 3
	switchoff 0
	
	code = 0
	pos1 = 127
	pos2 = 127
	pos3 = 127
	
	pause 5000
	
'Reset each of the 18X chips following this chip's restart

	switchon 5		'Switch the pitch rate to slow
	gosub InitPitch
	
	switchon 2		'Switch the roll rate to slow
	gosub InitRoll
	
	switchon portc 7	'Switch the heave rate to slow
	gosub InitHeave	
	

'Start of main loop

MainLoop:

'Check if any of the receiving 18X chips are requesting initialisation settings

	if Pin4 = 1 then gosub InitPitch
	
	if Pin3 = 1 then gosub InitRoll
	
	if Pin2 = 1 then gosub InitHeave

'Read the data from serial port into input0 at 4800 baud into bytes 1 to 3 - execution will pause
'here untill string "AB" is received and four 8 bit bytes are read. This effectively causes the chip to
'cycle at the rate the PC sends data

	serin 0, N4800, ("AB") , code, pos1, pos2, pos3
	
'If code indicates a re-set instruction for an 18X chip then go to the appropriate subroutine

TryAgain:

	if code = 6 then
		gosub RecPitch
		switchon 5		'Switch the pitch rate to slow
		pause 250
		gosub InitPitch
		goto MainLoop
	end if
	
	if code = 7 then
		gosub RecRoll
		switchon 2		'Switch the roll rate to slow
		pause 250
		gosub InitRoll
		goto MainLoop
	end if
	
	if code = 8 then
		gosub RecHeave
		switchon portc 7	'Switch the heave rate to slow
		pause 250
		gosub InitHeave
		goto MainLoop
	end if
	
'Otherwise calculate the required pulse lengths to set the normal position demands

	pulselength3 = pos3 * 1 MIN 0 MAX 255		'Heave demand
	pulselength3 = pulselength3 + 10
	
	pulselength1 = pos1 * 1 MIN 0 MAX 255	 	'for Pitch
	pulselength1 = pulselength1 + 10

	pulselength2 = pos2 * 1 MIN 0 MAX 255		'for Roll
	pulselength2 = pulselength2 + 10

	
'Write the data by sending pulses to the data pins - pulse length is proportional to position demand
'so pulse length is in range from 10 to 265 us*10 representing full right to full left DOF position
'ie pulse from 0.1 to 2.65 ms in length

'1st for the heave DOF

	if code = 4 or code = 5 then					'Set high/low speed flag
		switchon portc 7
	else
		switchoff portc 7
	end if
	
	if Pin2 = 1 then MainLoop					'If pin 2 is high 18X chip needs re-intialised
	
	switchon 0								'Set data ready pin high
	
	pulsin 2, 1, ready						'Wait for a pulse on pin 2 to transmit
	
	switchoff 0								'switch the data ready pin back off
	
	if ready = 0 then getout					'exit if excessive delay from receiving chip
	
	pulsout 1, pulselength3						'transmit
	
	 


'2nd for the pitch DOF

	if code = 2 or code = 5 then					'Set the motion rate pin
		switchon 5
	else
		switchoff 5
	end if
	
	if Pin4 = 1 then MainLoop
	
	switchon 6								'Set data ready pin high
	
	pulsin 4, 1, ready						'Wait for a pulse on pin 4 to transmit
	
	switchoff 6								'Switch data ready low
	
	if ready = 0 then getout					'exit
	
	pulsout 7, pulselength1						'transmit
	



'Last for the Roll DOF - this gets data last so sees most delay - but it still isn't much


	if code = 3 or code = 5 then
		switchon 2
	else
		switchoff 2
	end if
	
	if Pin3 = 1 then MainLoop
	
	switchon 3								'Set data ready pin high
	
	pulsin 3, 1, ready						'Wait for a pulse on pin 3 to transmit
	
	switchoff 3								'set data ready low
	
	if ready = 0 then getout					'exit
										
	pulsout 4, pulselength2						'transmit
	
	


'Return to the beginning to wait for next data from PC

	goto MainLoop


	End
	
	
	
	
'Subroutine to initialise the pitch DOF 18X

InitPitch:

'Calculate the pulselengths to send

	read 1, b7
	read 2, b6
	pulselength1 = w3 * 1 MIN 0 MAX 255	 	'for Pitch initial position demand
	pulselength1 = pulselength1 + 10

	read 3, b7
	read 4, b6
	pulselength2 = w3 * 10 MIN 0 MAX 25500 	'for cutoff value
	pulselength2 = pulselength2 + 10
	
	read 5, b7
	read 6, b6
	pulselength3 = w3 * 100 MIN 0 MAX 2550	'for threshhold value
	pulselength3 = pulselength3 + 10
		

'Now tell the 18X the data is ready
	
	switchon 6				'Set pitch data ready pin high

	do					'Wait for Pin4 to go low if it hasn't already done so
	
		b0 = 1
		if pin4 = 0 then
			b0 = 0
		end if
	
	loop while b0 = 1
	
'Send the 500 long pulse to indicate that setup data is coming

	pulsin 4, 1, ready				'Wait for a pulse on pin 4 to transmit
	
	switchoff 6						'Switch data ready back to low
	
	if ready = 0 then getout			'exit
	
	pulsout 7, 500					'transmit setup data coming signal
	
	
	
'1st set the initial position
	
	pulsin 4, 1, ready				'Wait for a pulse on pin 4 to transmit
	
	if ready = 0 then getout			'exit
	
	pulsout 7, pulselength1				'transmit initial position setting
	
	
'2nd set the cutoff
	
	pulsin 4, 1, ready				'Wait for a pulse on pin 4 to transmit
	
	if ready = 0 then getout			'exit
	
	pulsout 7, pulselength2				'transmit cutoff setting
	
	
'3rd set the voltage threshhold
	
	pulsin 4, 1, ready				'Wait for a pulse on pin 4 to transmit
	
	if ready = 0 then getout			'exit
	
	pulsout 7, pulselength3				'transmit the movement threshhold setting
	
'Finally wait for the confirmation pulse then return
	
	pulsin 4, 1, ready				'Wait for a pulse on pin 4 to transmit
	
	if ready = 0 then getout			'exit	
	
	pause 5000						'Wait to give time for the receiving chip to return
								'to normal processing 
	
'And return

return




'Subroutine to initialise the roll DOF 18X

InitRoll:

'Calculate the pulselengths to send


	read 7, b7
	read 8, b6
	pulselength1 = w3 * 1 MIN 0 MAX 255	 	'for Roll initial position demand
	pulselength1 = pulselength1 + 10

	read 9, b7
	read 10, b6
	pulselength2 = w3 * 10 MIN 0 MAX 25500 	'for cutoff value
	pulselength2 = pulselength2 + 10
	
	read 11, b7
	read 12, b6
	pulselength3 = w3 * 100 MIN 0 MAX 2550	'for threshhold value
	pulselength3 = pulselength3 + 10


'Now tell the 18X the data is ready
	
	switchon 3				'Set roll data ready pin high

	do					'Wait for Pin3 to go low if it hasn't already done so
	
		b0 = 1
		if pin3 = 0 then
			b0 = 0
		end if
	
	loop while b0 = 1
	
'Send the 500 long pulse to indicate that setup data is coming

	pulsin 3, 1, ready				'Wait for a pulse on pin 3 to transmit
	
	switchoff 3						'Switch data ready back to low
	
	if ready = 0 then getout			'exit
	
	pulsout 4, 500					'transmit setup data coming signal
	
	
	
'1st set the initial position
	
	pulsin 3, 1, ready				'Wait for a pulse on pin 3 to transmit
	
	if ready = 0 then getout			'exit
	
	pulsout 4, pulselength1				'transmit initial position setting
	
	
'2nd set the cutoff
	
	pulsin 3, 1, ready				'Wait for a pulse on pin 3 to transmit
	
	if ready = 0 then getout			'exit
	
	pulsout 4, pulselength2				'transmit cutoff setting
	
	
'3rd set the voltage threshhold
	
	pulsin 3, 1, ready				'Wait for a pulse on pin 3 to transmit
	
	if ready = 0 then getout			'exit
	
	pulsout 4, pulselength3				'transmit the movement threshhold setting
	
'Finally wait for the confirmation pulse then return
	
	pulsin 3, 1, ready				'Wait for a pulse on pin 3 to transmit
	
	if ready = 0 then getout			'exit	
	
	pause 5000
	
'And return

return





'Subroutine to initialise the Heave 18X

InitHeave:

'Calculate the pulselengths to send from stored data read from EEPROM

	read 13, b7
	read 14, b6
	pulselength1 = w3 * 1 MIN 0 MAX 255	 	'for Heave initial position demand
	pulselength1 = pulselength1 + 10

	read 15, b7
	read 16, b6
	pulselength2 = w3 * 10 MIN 0 MAX 25500 	'for cutoff value
	pulselength2 = pulselength2 + 10
	
	read 17, b7
	read 18, b6
	pulselength3 = w3 * 100 MIN 0 MAX 2550	'for threshhold value
	pulselength3 = pulselength3 + 10
	
'Now tell the 18X the data is ready
	
	switchon 0				'Set heave data ready pin high

	do					'Wait for Pin2 to go low if it hasn't already done so
	
		b0 = 1
		if pin2 = 0 then
			b0 = 0
		end if
	
	loop while b0 = 1
	
'Send the 500 long pulse to indicate that setup data is coming

	pulsin 2, 1, ready				'Wait for a pulse on pin 2 to transmit
	
	switchoff 0						'Switch data ready back to low
	
	if ready = 0 then getout			'exit
	
	pulsout 1, 500					'transmit setup data coming signal
	
	
	
'1st set the initial position
	
	pulsin 2, 1, ready				'Wait for a pulse on pin 2 to transmit
	
	if ready = 0 then getout			'exit
	
	pulsout 1, pulselength1				'transmit initial position setting
	
	
'2nd set the cutoff
	
	pulsin 2, 1, ready				'Wait for a pulse on pin 2 to transmit
	
	if ready = 0 then getout			'exit
	
	pulsout 1, pulselength2				'transmit cutoff setting
	
	
'3rd set the voltage threshhold
	
	pulsin 2, 1, ready				'Wait for a pulse on pin 2 to transmit
	
	if ready = 0 then getout			'exit
	
	pulsout 1, pulselength3				'transmit the movement threshhold setting
	
'Finally wait for the confirmation pulse then return
	
	pulsin 2, 1, ready				'Wait for a pulse on pin 2 to transmit
	
	if ready = 0 then getout			'exit	
	
	pause 5000
	
'And return

return




'Subroutine to store the received Pitch 18X settings in EEPROM

RecPitch:

	read 1, b7				'now write the new values to eeprom if they are different from
	read 2, b6				'the current settings
	w4 = pos1
	if w3 != w4 then
		w3 = w4
		write 1, b7
		write 2, b6
	end if
	
	read 3, b7
	read 4, b6
	w4 = pos2 * 10
	if w3 != w4 then
		w3 = w4
		write 3, b7
		write 4, b6
	end if	
	
	read 5, b7
	read 6, b6
	w4 = pos3
	if w3 != w4 then
		w3 = w4
		write 5, b7
		write 6, b6
	end if
	
return


'Subroutine to store the received Roll 18X settings in EEPROM

RecRoll:

	read 7, b7				'now write the new values to eeprom if they are different from
	read 8, b6				'the current settings
	w4 = pos1
	if w3 != w4 then
		w3 = w4
		write 7, b7
		write 8, b6
	end if
	
	read 9, b7
	read 10, b6
	w4 = pos2 * 10
	if w3 != w4 then
		w3 = w4
		write 9, b7
		write 10, b6
	end if	
	
	read 11, b7
	read 12, b6
	w4 = pos3
	if w3 != w4 then
		w3 = w4
		write 11, b7
		write 12, b6
	end if
	
return




'Subroutine to store the received Heave 18X settings in EEPROM

RecHeave:

	read 13, b7				'write the new values to eeprom if they are different from
	read 14, b6				'the current settings
	w4 = pos1
	if w3 != w4 then
		w3 = w4
		write 13, b7
		write 14, b6
	end if
	
	read 15, b7
	read 16, b6
	w4 = pos2 * 10
	if w3 != w4 then
		w3 = w4
		write 15, b7
		write 16, b6
	end if	
	
	read 17, b7
	read 18, b6
	w4 = pos3
	if w3 != w4 then
		w3 = w4
		write 17, b7
		write 18, b6
	end if
	
return


'Exit if any pulsin times out
getout:

'wait for a serin but only act if it is to reset a chip - don't pass normal position demand through

	
	serin 0, N4800, ("AB") , code, pos1, pos2, pos3
	
'If code indicates a re-set instruction for an 18X chip then go to the appropriate subroutine

	if code > 5 and code < 9 then TryAgain
	
'Stay in this loop until a chip reset instruction is received
	
	goto getout
